<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class PermissionTableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $permissions = [
            'dashboard.view',
            'users.view',
            'users.create',
            'users.edit',
            'users.delete',
            'users.toggle_status',
            'users.change_password',
            'roles.view',
            'roles.create',
            'roles.edit',
            'roles.delete',
            'roles.assign',
            'wards.view',
            'wards.create',
            'wards.edit',
            'wards.delete',
            'zones.view',
            'zones.create',
            'zones.edit',
            'zones.delete',
            'approval-request-hierarchies.view',
            'approval-request-hierarchies.create',
            'approval-request-hierarchies.edit',
            'approval-request-hierarchies.delete',
            'fees.view',
            'fees.create',
            'fees.edit',
            'fees.delete',
            'signature.view',
            'signature.create',
            'signature.edit',
            'signature.delete',
            'nature-business.view',
            'nature-business.create',
            'nature-business.edit',
            'nature-business.delete',

            // Applications
            'application.pending-list',
            'application.approval-list',
            'application.rejected-list',
            'application.my-application',
            'application.track-application',
            'application.verify_offline_reciept',

            // User Profile
            'users.user-profile',

            // Other
            'other.rts-guideline',
            'other.important-link',
            'other.contact',

            // Admin Extra Links
            'mandap-entry.create',
            'report.marriage-report',
            'report.services-report',
        ];

        // Create / Update permissions
        foreach ($permissions as $permissionName) {
            Permission::firstOrCreate(
                ['name' => $permissionName],
                ['group' => explode('.', $permissionName)[0]]
            );
        }

        $this->command->info('All missing permissions have been added!');

        // Assign permissions to roles
        $this->assignPermissionsToRoles();
    }

    /**
     * Assign permissions to roles
     */
    protected function assignPermissionsToRoles()
    {
        // Create roles if not exist
        $superAdminRole = Role::firstOrCreate(['name' => 'Super Admin']);
        $userRole = Role::firstOrCreate(['name' => 'User']);
        $otherRole = Role::firstOrCreate(['name' => 'Admin']); // or any other role

        // Super Admin gets ALL permissions
        $allPermissions = Permission::all();
        $superAdminRole->syncPermissions($allPermissions);

        // Regular User permissions
        $userPermissions = Permission::whereIn('name', [
            'other.rts-guideline',
            'other.important-link',
            'other.contact',
            'application.track-application',
            'application.my-application',
            'dashboard.view',
             'users.user-profile',
        ])->get();
        $userRole->syncPermissions($userPermissions);

        // Other roles (Admin) permissions
        $otherPermissions = Permission::whereIn('name', [
            'dashboard.view',
            'application.pending-list',
            'application.approval-list',
            'application.rejected-list',
            'application.track-application',
            'application.verify_offline_reciept',
        ])->get();
        // Assign remaining permissions to all roles except Super Admin and User
        Role::whereNotIn('name', ['Super Admin', 'User'])->get()->each(function ($role) use ($otherPermissions) {
            $role->syncPermissions($otherPermissions);
        });
        $this->command->info('Permissions assigned to Super Admin, User, and Other roles.');
    }
}
