(function() {

    window.attachGlobalValidation = function(formSelector, saveBtnSelector, options = {}) {
        const form = document.querySelector(formSelector);
        const saveBtn = document.querySelector(saveBtnSelector);
        if (!form || !saveBtn) return;

        const containerId = options.containerId || 'addContainer';
        const printTitleText = options.printTitle || 'Trade License';

        // Robust visibility check
        function isVisible(el) {
            if (!el) return false;
            if (el.hidden) return false;
            const style = window.getComputedStyle(el);
            if (style.display === 'none' || style.visibility === 'hidden' || style.opacity === '0') return false;
            // offsetParent covers many cases (excluded for fixed elements), also check clientRects
            if (el.offsetParent === null && el.getClientRects().length === 0) return false;
            return true;
        }

        // Get only visible required fields (at time of validation)
        function getVisibleRequiredFields() {
            return Array.from(form.querySelectorAll('[required]')).filter(f => {
                // if any ancestor is not visible, treat as hidden
                let node = f;
                while (node && node !== document) {
                    if (!isVisible(node)) return false;
                    node = node.parentElement;
                }
                return true;
            });
        }

        function findOrCreateErrorSpan(field) {
            const fieldId = field.id;

            // 1. Check for a specific error span using the field ID (e.g., .aadhar_num_err)
            const specificErr = field.parentElement.querySelector(`.${fieldId}_err`);
            if (specificErr) {
                // Ensure it also has the text-danger class for visibility/styling,
                // and add 'required-error' if it's not a generic error span already.
                specificErr.classList.add('text-danger', 'required-error');
                return specificErr;
            }

            // 2. Fallback to check the immediate next sibling for a generic error span
            let el = field.nextElementSibling;
            if (el && el.classList && el.classList.contains('text-danger') && el.classList.contains('required-error')) {
                return el;
            }

            // 3. Create a new general error span if none were found
            const span = document.createElement('span');
            span.className = 'text-danger required-error';
            if (fieldId === 'is_correct_info') {
                const parentCol = field.closest('.col-md-12');
                const formCheckDiv = field.closest('.form-check'); // The div containing the checkbox and text

                // Check if the custom error span already exists right after the form-check div
                let errSpan = formCheckDiv.nextElementSibling;
                if (errSpan && errSpan.classList.contains(`${fieldId}_err`)) {
                    errSpan.classList.add('text-danger', 'required-error');
                    return errSpan;
                }

                // Create a new general error span and insert it after the form-check div
                const span = document.createElement('span');
                span.className = 'text-danger required-error';
                span.classList.add(`${fieldId}_err`); // Add the specific error class
                span.style.fontSize = '13px';
                span.style.marginTop = '4px'; // Add a small margin above the error text
                span.style.display = 'none'; // Initially hidden

                // Insert it immediately after the form-check div (which contains the checkbox and text)
                formCheckDiv.insertAdjacentElement('afterend', span);
                return span;
            }
            span.style.fontSize = '13px';
            field.insertAdjacentElement('afterend', span);
            return span;
        }


        // Show all general required field errors at once
        function showAllErrors() {
            let allFilled = true;
            getVisibleRequiredFields().forEach(field => {
                if (['mobile_no', 'aadhar_num', 'email', 'pincode'].includes(field.id)) return;

                const val = (field.value || '').toString().trim();
                const notFilled = (field.type === 'file' && field.files.length === 0) ||
                    (field.type === 'checkbox' && !field.checked) ||
                    (val === '');

                // Use findOrCreateErrorSpan, which handles the specific mobile_no_err span
                const errorSpan = findOrCreateErrorSpan(field);

                // Clear any previous mobile-specific error before checking general required status
                if (errorSpan.classList.contains('mobile_no_err') && errorSpan.textContent.includes('10 digits')) {
                    errorSpan.textContent = '';
                }

                if (notFilled) {
                    errorSpan.textContent = 'This field is required';
                    errorSpan.style.display = 'block';

                    // if (field.id === 'is_correct_info') {
                    //     errorSpan.style.display = 'block'; // Ensure it's a block element
                    //     errorSpan.style.marginTop = '5px'; // Add some spacing below the text
                    // }

                    allFilled = false;
                } else {
                    // Only clear if the error is a 'required' error, not a specific mobile validation error
                    if (errorSpan.textContent === 'This field is required' || errorSpan.textContent === '') {
                        errorSpan.textContent = '';
                        errorSpan.style.display = 'none';
                    }
                }
            });
            return allFilled;
        }

        /**
         * 🚨 NEW FUNCTION: Specific Mobile Number Validation 🚨
         * Checks if the mobile number is present and is exactly 10 digits.
         * @returns {boolean} True if validation passes, false otherwise.
         */
        function validateMobileNumber() {
            const mobileField = form.querySelector('#mobile_no');
            if (!mobileField || !isVisible(mobileField)) return true; // Skip if not found or not visible

            const mobileValue = mobileField.value.trim();
            const errorSpan = findOrCreateErrorSpan(mobileField);
            const mobileRegex = /^\d{10}$/; // Exactly 10 digits

            // // 1. Check if the field is empty (This is handled by showAllErrors but good for clarity)
            if (mobileField.required && mobileValue === '') {
                errorSpan.textContent = 'This field is required'; // 🛑 Show the required error
                errorSpan.style.display = 'block';
                return false; // 🛑 Fail the validation
            }

            // 2. Check for 10 digits using regex
            if (mobileValue !== '' && !mobileRegex.test(mobileValue)) {
                errorSpan.textContent = 'Mobile number must be exactly 10 digits.';
                errorSpan.style.display = 'block';
                return false;
            } else {
                // Clear error if it passes, but only if the error text was the specific mobile error
                if (errorSpan.textContent.includes('10 digits') || errorSpan.textContent === '') {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
                return true;
            }
        }

        function validateAadharNumber() {
            const aadharField = form.querySelector('#aadhar_num');
            if (!aadharField || !isVisible(aadharField)) return true;

            const aadharValue = aadharField.value.trim();
            const errorSpan = findOrCreateErrorSpan(aadharField);
            const aadharRegex = /^\d{12}$/; // Exactly 12 digits

            if (aadharField.required && aadharValue === '') {
                errorSpan.textContent = 'This field is required'; // 🛑 Show the required error
                errorSpan.style.display = 'block';
                return false; // 🛑 Fail the validation
            }

            if (aadharValue !== '' && !aadharRegex.test(aadharValue)) {
                errorSpan.textContent = 'Aadhar number must be exactly 12 digits.';
                errorSpan.style.display = 'block';
                return false;
            } else {
                // Clear error if it passes, but only if the error text was the specific aadhar error
                if (errorSpan.textContent.includes('12 digits') || errorSpan.textContent === '') {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
                return true;
            }
        }

        function validateEmail() {
            const emailField = form.querySelector('#email');
            if (!emailField || !isVisible(emailField)) return true;

            const emailValue = emailField.value.trim();
            const errorSpan = findOrCreateErrorSpan(emailField);
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]{2,}$/;

            if (emailField.required && emailValue === '') {
                errorSpan.textContent = 'This field is required'; // 🛑 Show the required error
                errorSpan.style.display = 'block';
                return false; // 🛑 Fail the validation
            }

            if (emailValue !== '' && !emailRegex.test(emailValue)) {
                errorSpan.textContent = 'Please enter a valid email address.';
                errorSpan.style.display = 'block';
                return false;
            } else {
                // Clear error if it passes, but only if the error text was the specific email error
                if (errorSpan.textContent.includes('valid email address') || errorSpan.textContent === '') {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
                return true;
            }
        }

        function validatePincode() {
            const pincodeField = form.querySelector('#pincode');
            if (!pincodeField || !isVisible(pincodeField)) return true;

            const pincodeValue = pincodeField.value.trim();
            const errorSpan = findOrCreateErrorSpan(pincodeField);
            const pincodeRegex = /^\d{6}$/; // Exactly 6 digits

            // 1. Check for required field
            if (pincodeField.required && pincodeValue === '') {
                errorSpan.textContent = 'This field is required';
                errorSpan.style.display = 'block';
                return false;
            }

            // 2. Check for 6 digits
            if (pincodeValue !== '' && !pincodeRegex.test(pincodeValue)) {
                errorSpan.textContent = 'Pincode must be exactly 6 digits.';
                errorSpan.style.display = 'block';
                return false;
            } else {
                // Clear error if it passes, but only if the error text was the specific pincode error
                if (errorSpan.textContent.includes('6 digits') || errorSpan.textContent === '') {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
                return true;
            }
        }

        // Clear error dynamically for visible fields only
        function clearFieldError(e) {
            const field = e.target;
            if (!field.required && !['mobile_no', 'aadhar_num', 'email', 'pincode'].includes(field.id)) return;
            if (!isVisible(field)) return;
            const val = (field.value || '').toString().trim();
            const filled = (field.type === 'file' && field.files.length > 0) ||
                (field.type === 'checkbox' && field.checked) ||
                (val !== '');

            const errorSpan = findOrCreateErrorSpan(field);

            // Clear general required error
            if (errorSpan && filled && errorSpan.textContent === 'This field is required') {
                errorSpan.textContent = '';
                errorSpan.style.display = 'none';
            }

            // Clear mobile-specific error on input as the user types
            if (field.id === 'mobile_no' && errorSpan && errorSpan.textContent.includes('10 digits')) {
                const mobileRegex = /^\d{10}$/;
                if (mobileRegex.test(val)) {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
            }
            if (field.id === 'aadhar_num' && errorSpan && errorSpan.textContent.includes('12 digits')) {
                const aadharRegex = /^\d{12}$/;
                if (aadharRegex.test(val)) {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
            }

            // 🚨 Clear Email-specific error on input as the user types
            if (field.id === 'email' && errorSpan && errorSpan.textContent.includes('valid email address')) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]{2,}$/;
                if (emailRegex.test(val)) {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
            }
            if (field.id === 'pincode' && errorSpan && errorSpan.textContent.includes('6 digits')) {
                const pincodeRegex = /^\d{6}$/;
                if (pincodeRegex.test(val)) {
                    errorSpan.textContent = '';
                    errorSpan.style.display = 'none';
                }
            }
        }

        form.addEventListener('input', clearFieldError, true);
        form.addEventListener('change', clearFieldError, true);

        // On Save & Print click
        saveBtn.addEventListener('click', function(e) {
            e.preventDefault();

            // 1. Run Mobile Number Validation FIRST
            const mobileValid = validateMobileNumber();
            const aadharValid = validateAadharNumber(); // 🚨 NEW
            const emailValid = validateEmail(); // 🚨 NEW
            const pincodeValid = validatePincode();

            const generalValid = showAllErrors();
            const allValid = mobileValid && aadharValid && emailValid && generalValid && pincodeValid; // 🛑 Combine ALL results

            if (!allValid) {
                // Now we search for the FIRST error, regardless of which group it belongs to.
                // It will be either a specific error (mobile/aadhar/email) OR a general error.
                const firstErr = form.querySelector('.text-danger:not(:empty)');

                if (firstErr) {
                    firstErr.scrollIntoView({ behavior: 'smooth', block: 'center' });
                }
                return; // 🛑 Only return once, after checking EVERYTHING
            }

            // After all validation passes, proceed with form logic...
            // ... (rest of the original code follows) ...

            // After validation passes, reveal upload / footer and hide Save & Print
            const uploadSection = document.getElementById('upload_section');
            if (uploadSection) uploadSection.style.display = 'block';
            const cardFooter = document.getElementById('card-footer');
            if (cardFooter) cardFooter.style.display = 'block';
            saveBtn.style.display = 'none'; // hide Save & Print after making upload visible

            const addContainer = document.getElementById(containerId) || form.querySelector(`#${containerId}`);
            if (!addContainer) return;

            // --- PDF Generation ---
            const clone = addContainer.cloneNode(true);
            clone.style.cssText = `
                font-size:16px; line-height:1.4; padding:20px; margin:0 auto;
                background:#fff; width:210mm; min-height:297mm; box-sizing:border-box;
            `;

            const uploadLabel = clone.querySelector('#upload_section');
            if (uploadLabel) {
                uploadLabel.remove();
            }

            const originalSelects = form.querySelectorAll('select');
            Array.from(originalSelects).forEach(originalSelect => {

                let cloneSelect = clone.querySelector(`#${originalSelect.id}`);
                if (!cloneSelect) {
                    cloneSelect = clone.querySelector(`[name="${originalSelect.name}"]`);
                }
                if (!cloneSelect) return;

                const selectedIndex = originalSelect.selectedIndex;

                if (selectedIndex > 0) {
                    const selectedText = originalSelect.options[selectedIndex].textContent.trim();
                    const parentDiv = cloneSelect.parentElement;

                    const newDisplay = document.createElement('div');
                    newDisplay.textContent = selectedText;

                    newDisplay.style.cssText = `
                                padding: 8px 12px;
                                border: 1px solid;
                                border-radius: 4px;
                                background: #fff;
                                min-height: 34px;
                                display: flex;
                                align-items: center;
                                margin: 4px 0;
                            `;

                    parentDiv.appendChild(newDisplay);
                }

                cloneSelect.remove();
            });

            Array.from(form.querySelectorAll('input[type="file"]')).forEach(fileInput => {
                const cloneFileInput = clone.querySelector(`#${fileInput.id}`);

                if (fileInput.files.length > 0 && cloneFileInput) {
                    const fileName = fileInput.files[0].name;
                    const parentColDiv = cloneFileInput.parentElement;

                    if (parentColDiv) {
                        const labelEl = parentColDiv.querySelector('.col-form-label');
                        const labelText = labelEl ? labelEl.textContent.replace('*', '').trim() : 'Uploaded Document';
                        if (labelEl) labelEl.remove();

                        const wrapperDiv = document.createElement('div');
                        wrapperDiv.style.cssText = `
                                margin: 10px 0;
                                width: 100%;
                                display: flex;
                                flex-direction: column;
                            `;

                        const titleDiv = document.createElement('div');
                        titleDiv.textContent = labelText;
                        titleDiv.style.cssText = `
                                font-weight: bold;
                                margin-bottom: 4px;
                            `;

                        const boxDiv = document.createElement('div');
                        boxDiv.textContent = fileName;
                        boxDiv.style.cssText = `
                            border: 2px solid #000;
                            border-radius: 6px;
                            padding: 8px 10px;
                            background: #fff;
                            min-height: 40px;
                            width: 100%;
                            max-width: 250px; /* Adjust if needed */
                            word-wrap: break-word;
                            overflow-wrap: break-word;
                            white-space: normal;
                            line-height: 1.3;
                            box-sizing: border-box;
                            font-weight: 500;
                            `;

                        wrapperDiv.appendChild(titleDiv);
                        wrapperDiv.appendChild(boxDiv);

                        parentColDiv.appendChild(wrapperDiv);
                    }
                }

                if (cloneFileInput) cloneFileInput.remove();
            });

            Array.from(form.querySelectorAll('input[type="date"]')).forEach(originalDateInput => {
                const cloneDateInput = clone.querySelector(`#${originalDateInput.id}`);

                if (cloneDateInput) {
                    const dateValue = originalDateInput.value; // YYYY-MM-DD

                    if (dateValue) {
                        const [year, month, day] = dateValue.split('-');
                        const formattedDate = `${day}-${month}-${year}`;

                        // ✅ Create replacement element that looks like input box
                        const newBox = document.createElement('div');
                        newBox.textContent = formattedDate;
                        newBox.style.border = "1px solid #000";
                        newBox.style.padding = "6px 10px";
                        newBox.style.minHeight = "32px";
                        newBox.style.borderRadius = "4px";
                        newBox.style.fontSize = "14px";
                        newBox.style.display = "flex";
                        newBox.style.alignItems = "center";

                        cloneDateInput.insertAdjacentElement('afterend', newBox);
                    }

                    // ✅ Remove input element only
                    cloneDateInput.remove();
                }
            });

            Array.from(clone.querySelectorAll('*')).forEach(el => {
                el.style.fontSize = '16px';
                el.style.margin = '4px 0';
            });

            const printTitle = document.createElement('h2');
            printTitle.style.textAlign = 'center';
            printTitle.style.marginBottom = '18px';
            printTitle.style.fontSize = '18px';
            printTitle.textContent = printTitleText;
            clone.insertBefore(printTitle, clone.firstChild);

            Array.from(clone.querySelectorAll('button')).forEach(btn => btn.remove());
            Array.from(clone.querySelectorAll('.card')).forEach(card => {
                card.style.background = 'white';
                card.style.boxShadow = 'none';
                card.style.border = '1px solid #ccc';
                card.style.marginBottom = '12px';
                card.style.padding = '12px';
            });
            Array.from(clone.querySelectorAll('.selected-value')).forEach(div => div.classList.remove('d-none'));

            let printContainer = document.getElementById('printContainer');
            if (!printContainer) {
                printContainer = document.createElement('div');
                printContainer.id = 'printContainer';
                printContainer.style.position = 'fixed';
                printContainer.style.left = '-9999px';
                document.body.appendChild(printContainer);
            }
            printContainer.innerHTML = '';
            printContainer.appendChild(clone);

            html2canvas(clone, { scrollY: -window.scrollY, backgroundColor: '#fff', scale: 1, useCORS: true })
                .then(canvas => {
                    const imgData = canvas.toDataURL('image/jpeg', 0.6);
                    const { jsPDF } = window.jspdf;
                    const pdf = new jsPDF('p', 'mm', 'a4');
                    const margin = 10;
                    pdf.addImage(
                        imgData,
                        'JPEG',
                        margin,
                        margin,
                        pdf.internal.pageSize.getWidth() - margin * 2,
                        pdf.internal.pageSize.getHeight() - margin * 2
                    );
                    pdf.save(`${printTitleText}.pdf`);
                })
                .catch(err => {
                    console.error('PDF generation failed', err);
                    alert('Unable to generate PDF. Please try again.');
                });
        });
    };

})();
