
        document.addEventListener('DOMContentLoaded', () => {

        // --- Detect logical field type (only for text-like inputs) ---
        function detectFieldType(input) {
        // Ignore non-text-like input types (e.g., file upload)
        if (!['text', 'email', 'number', 'tel', 'password'].includes(input.type)) {
            return null;
        }

        const s = ((input.name || '') + ' ' + (input.id || '') + ' ' + (input.placeholder || '')).toLowerCase();

        if (/\baadhaar|\baadhar|\baadhar_no|\baadhar_num|\baadhar-number/.test(s)) return 'aadhar';
        if (/\bmobile|\bmobile_no|\bcontact|\bcontact_no|\bphone|\btelephone|\btel/.test(s)) return 'phone';
        if (/\bemail|\be-mail|\bmail|\balternet_email|\balternate_email|\bemail_id/.test(s)) return 'email';
        return null;
        }

        // --- Get associated error span ---
        function getErrorSpan(input) {
        if (input.name) {
            const spanByName = document.querySelector(`.${input.name}_err`);
            if (spanByName) return spanByName;
        }
        if (input.id) {
            const spanById = document.querySelector(`.${input.id}_err`);
            if (spanById) return spanById;
        }
        return input.closest('div')?.querySelector('span.text-danger') || null;
        }

        // --- Validation rules for specific text formats ---
        function validateValueByType(value, type, required) {
        if (type === 'aadhar') {
            if (required && value === '') return 'Aadhaar number is required';
            if (value && !/^\d{12}$/.test(value)) return 'Enter a valid 12-digit Aadhaar number';
            return '';
        }
        if (type === 'phone') {
            if (required && value === '') return 'Mobile number is required';
            if (value && !/^[6-9]\d{9}$/.test(value)) return 'Enter a valid 10-digit mobile number';
            return '';
        }
        if (type === 'email') {
            if (required && value === '') return 'Email is required';
            if (value && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) return 'Enter a valid email address';
            return '';
        }
        return '';
        }

        // --- Validate a single input field ---
        function validateField(input) {
        // Skip all non-text inputs (file, checkbox, radio, hidden)
        if (!['text', 'email', 'number', 'tel', 'password'].includes(input.type)) return true;

        const type = detectFieldType(input);
        const required = input.hasAttribute('required');
        const value = (input.value || '').trim();

        let err = '';
        if (type) {
            err = validateValueByType(value, type, required);
        } else if (required && value === '') {
            err = 'This field is required';
        }

        const span = getErrorSpan(input);
        if (span) {
            span.textContent = err;
            span.style.display = err ? 'block' : 'none';
        }

        if (err) {
            input.classList.add('is-invalid');
            input.setAttribute('aria-invalid', 'true');
        } else {
            input.classList.remove('is-invalid');
            input.removeAttribute('aria-invalid');
        }

        return !err;
        }

        // --- Validate all visible text fields ---
        function validateAllFields(form) {
        let allValid = true;
        const inputs = Array.from(form.querySelectorAll('input, textarea'));
        inputs.forEach(input => {
            // Skip invisible or non-text inputs
            if (input.offsetParent === null) return;
            if (!['text', 'email', 'number', 'tel', 'password'].includes(input.type)) return;
            if (!validateField(input)) allValid = false;
        });
        return allValid;
        }

        // --- Attach validation handlers ---
        const allInputs = Array.from(document.querySelectorAll('input, textarea'));
        allInputs.forEach(input => {
        if (!['text', 'email', 'number', 'tel', 'password'].includes(input.type)) return;
        input.addEventListener('input', () => validateField(input));
        input.addEventListener('blur', () => validateField(input));
        });

        // --- Expose for use in global form validation ---
        window.validateAllFields = validateAllFields;

        });
